<?php

namespace MyTelegramBot;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;

/**
 * A simple and modern PHP library for interacting with the Telegram Bot API.
 */
class TelegramBot
{
    private string $botToken;
    private Client $httpClient;
    private string $apiUrl;

    /**
     * Initialize the Telegram bot with a token.
     *
     * @param string $botToken Telegram Bot API token
     * @throws \InvalidArgumentException If bot token is empty
     */
    public function __construct(string $botToken)
    {
        if (empty($botToken)) {
            throw new \InvalidArgumentException('Bot token cannot be empty.');
        }

        $this->botToken = $botToken;
        $this->apiUrl = "https://api.telegram.org/bot{$botToken}/";
        $this->httpClient = new Client(['base_uri' => $this->apiUrl]);
    }

    /**
     * Send a request to the Telegram API.
     *
     * @param string $method API method name (e.g., sendMessage)
     * @param array $params Parameters for the request
     * @return array Telegram API response
     * @throws \RuntimeException If the request fails or API returns an error
     */
    private function sendRequest(string $method, array $params = []): array
    {
        try {
            $response = $this->httpClient->post($method, ['form_params' => $params]);
            $result = json_decode($response->getBody()->getContents(), true);

            if (!$result['ok']) {
                throw new \RuntimeException("Telegram API error: {$result['description']}");
            }

            return $result['result'];
        } catch (GuzzleException $e) {
            throw new \RuntimeException("Failed to send request to Telegram API: {$e->getMessage()}");
        }
    }

    /**
     * Send a text message to a chat.
     *
     * @param int|string $chatId Unique identifier for the target chat
     * @param string $text Message text to send
     * @return array Telegram API response
     */
    public function sendMessage($chatId, string $text): array
    {
        $params = [
            'chat_id' => $chatId,
            'text' => $text,
        ];

        return $this->sendRequest('sendMessage', $params);
    }

    /**
     * Set a webhook for the bot.
     *
     * @param string $webhookUrl HTTPS URL to send updates to
     * @return array Telegram API response
     * @throws \InvalidArgumentException If webhook URL is invalid
     */
    public function setWebhook(string $webhookUrl): array
    {
        if (!filter_var($webhookUrl, FILTER_VALIDATE_URL) || strpos($webhookUrl, 'https://') !== 0) {
            throw new \InvalidArgumentException('Webhook URL must be a valid HTTPS URL.');
        }

        $params = ['url' => $webhookUrl];
        return $this->sendRequest('setWebhook', $params);
    }
}
?>